<?php

namespace App\Http\Controllers\Dashboard\Configuraciones;

use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use Stevebauman\Location\Facades\Location;

class RolesController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function logRegistro($request, $status, $rol, $mensaje)
    {
        $permisos = $rol->permissions == null ? $request->permisos : $rol->permissions->pluck('name');
        $rolName = $rol->name == null ? $request->nombre_rol : $rol->name;

        $arrayDatos = [
            'fecha/hora' => date('Y-m-d H:i:s'),
            'username' => session('nombreUsuarioSession'),
            'user_agent' => $request->header('User-Agent'),
            'url' => $request->fullUrl() . ' [' . $request->method() . ']',
            'location' => Location::get($request->ip()),
            'menseje' => $mensaje,
            'nombre_rol' => $rolName,
            'permisos' => $permisos
        ];

        if ($status == 'ok') {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->info("Registro de rol exitoso [" . $rolName . "]");
        } else {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->warning("Registro de rol fallido [" . $rolName . "]");
        }
    }

    public function logActulizacion($request, $status, $rol, $mensaje)
    {
        $permisos = $rol->permissions == null ? $request->permisos_edit : $rol->permissions->pluck('name');
        $rolName = $rol->name == null ? $request->nombre_rol_edit : $rol->name;

        $arrayDatos = [
            'fecha/hora' => date('Y-m-d H:i:s'),
            'username' => session('nombreUsuarioSession'),
            'user_agent' => $request->header('User-Agent'),
            'url' => $request->fullUrl() . ' [' . $request->method() . ']',
            'location' => Location::get($request->ip()),
            'menseje' => $mensaje,
            'nombre_rol' => $rolName,
            'permisos' => $permisos
        ];

        if ($status == 'ok') {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->info("Actualizacion de datos del rol exitoso [" . $rolName . "]");
        } else {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->warning("Actualizacion de datos del rol fallido [" . $rolName . "]");
        }
    }

    public function logEliminacion($request, $status, $mensaje)
    {
        $rolName = $request->nombre_rol_destroy;

        $arrayDatos = [
            'fecha/hora' => date('Y-m-d H:i:s'),
            'username' => session('nombreUsuarioSession'),
            'user_agent' => $request->header('User-Agent'),
            'url' => $request->fullUrl() . ' [' . $request->method() . ']',
            'location' => Location::get($request->ip()),
            'menseje' => $mensaje,
            'nombre_rol' => $rolName
        ];

        if ($status == 'ok') {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->info("Eliminacion de rol exitosa [" . $rolName . "]");
        } else {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->warning("Eliminacion de rol fallida [" . $rolName . "]");
        }
    }

    public function show()
    {
        return view('dashboard.configuraciones.roles');
    }

    public function rolesData()
    {
        $data = Role::all();

        return DataTables::of($data)
            ->addColumn('action', function ($role) {
                return view('layouts.dashboard.partials.roles.botones', compact('role'));
            })->make(true);
    }

    public function roleData($id)
    {
        //Se obtiene los datos del rol
        $roleData = Role::findOrFail($id);

        //Se obtiene los permisos del rol
        $permisos = DB::table('role_has_permissions')->where('role_id', $id)->get();

        //Retorna los datos del rol y los permisos en formato json
        return Response()->json(["success" => true, 'data' => $roleData, 'permisos' => $permisos]);
    }

    public function permissionData()
    {
        //Recuperamos todos los permisos en el sistema
        $permisos = DB::table('permissions')
            ->join('c_categorias_permisos', 'permissions.id_category', '=', 'c_categorias_permisos.id')
            ->join('c_sub_categorias_permisos', 'permissions.id_sub_category', '=', 'c_sub_categorias_permisos.id')
            ->orderBy('permissions.id', 'ASC')
            ->orderBy('c_categorias_permisos.categoria', 'ASC')
            ->select('permissions.id', 'permissions.description', 'permissions.id_category', 'permissions.id_sub_category', 'c_categorias_permisos.categoria', 'c_sub_categorias_permisos.sub_categoria')
            ->get();

        return Response()->json(["success" => true, 'data' => $permisos]);
    }

    public function registraRol(Request $request)
    {
        $validaNombreRol = Role::where('name', $request->nombre_rol)->count();

        if ($validaNombreRol >= 1) {
            $mensaje = 'El nombre del rol ya se encuentra registrado';
            $this->logRegistro($request, 'error', $request, $mensaje);
            notify()->error($mensaje, 'Registro del rol');
        } else {
            $rol = Role::create(['name' => strtoupper($request->nombre_rol)]);
            $rol->permissions()->sync($request->permisos);

            $mensaje = 'Rol registrado correctamente';
            $this->logRegistro($request, 'ok', $rol, $mensaje);
            notify()->success($mensaje, 'Registro del rol');
        }

        return redirect()->route('configuraciones.roles');
    }

    public function actulizaRol(Request $request)
    {
        //Recuperamos el usuario a actualizar
        $id = $request->id_rol_edit;

        //Se obtiene el nombre del rol y se transforma en mayuscula
        $nombreRol = strtoupper($request->nombre_rol_edit);

        //Verifica si el rol ya esta registrado
        $validaNombreRol = Role::where('name', $nombreRol)->where('id', '!=', $id)->count();

        if ($validaNombreRol >= 1) {
            $mensaje = 'El rol ya se encuentra registrado';
            $this->logActulizacion($request, 'error', $request, $mensaje);
            notify()->error($mensaje, 'Actualización del rol');
        } else {
            //Recuperamos el rol a actualizar
            $rol = Role::findOrFail($id);

            $rol->name = $nombreRol;
            $rol->update();
            $rol->permissions()->sync($request->permisos_edit);

            $mensaje = 'Rol actualizado correctamente';
            $this->logActulizacion($request, 'ok', $rol, $mensaje);
            notify()->success($mensaje, 'Actualización del rol');
        }

        return redirect()->route('configuraciones.roles');
    }

    public function eliminaRol(Request $request)
    {
        $id = $request->id_rol_destroy;

        //Valida si el rol no esta asignado a ningun usuario
        $validaRolUsuario = DB::table('model_has_roles')->where('role_id', '=', $id)->count();

        if ($validaRolUsuario >= 1) {
            $mensaje = 'El rol no puede ser eliminado porque esta asignado a un usuario';
            $this->logEliminacion($request, 'error', $mensaje);
            notify()->error($mensaje, 'Eliminación del rol');
        } else {
            //Recuperamos el rol a eliminar
            $rol = Role::findOrFail($id);
            $rol->delete();

            $mensaje = 'Rol eliminado correctamente';
            $this->logEliminacion($request, 'ok', $mensaje);
            notify()->success($mensaje, 'Eliminacion del rol');
        }

        return redirect()->route('configuraciones.roles');
    }
}
