<?php

namespace App\Http\Controllers\Dashboard\Configuraciones;

use App\Models\c_empresa;
use App\Models\Sucursale;
use App\Models\c_sucursale;
use Illuminate\Http\Request;
use Yajra\DataTables\DataTables;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Http\Controllers\Controller;
use App\Models\NumeroReceta;
use Illuminate\Support\Facades\Schema;
use Illuminate\Database\Schema\Blueprint;
use Stevebauman\Location\Facades\Location;

class SucursalesController extends Controller
{
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function logRegistro($request, $status, $mensaje)
    {
        $arrayDatos = [
            'fecha/hora' => date('Y-m-d H:i:s'),
            'username' => session('nombreUsuarioSession'),
            'user_agent' => $request->header('User-Agent'),
            'url' => $request->fullUrl() . ' [' . $request->method() . ']',
            'location' => Location::get($request->ip()),
            'menseje' => $mensaje,
            'datos_permiso' => $request->all()
        ];

        if ($status == 'ok') {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->info("Registro de permiso del sistema exitoso");
        } else {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->warning("Registro de permiso del sistema fallido");
        }
    }

    public function logActulizacion($request, $status, $permiso, $mensaje)
    {
        $datosPermiso = $permiso == null ? $request->all() : $permiso;

        $arrayDatos = [
            'fecha/hora' => date('Y-m-d H:i:s'),
            'username' => session('nombreUsuarioSession'),
            'user_agent' => $request->header('User-Agent'),
            'url' => $request->fullUrl() . ' [' . $request->method() . ']',
            'location' => Location::get($request->ip()),
            'menseje' => $mensaje,
            'datos_permisos' => $datosPermiso
        ];

        if ($status == 'ok') {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->info("Actualizacion de permiso del sistema exitoso");
        } else {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->warning("Actualizacion de permiso del sistema fallido");
        }
    }

    public function logEliminacion($request, $status, $permiso, $mensaje)
    {
        $datosPermiso = $permiso == null ? $request->id_permiso_destroy : $permiso;

        $arrayDatos = [
            'fecha/hora' => date('Y-m-d H:i:s'),
            'username' => session('nombreUsuarioSession'),
            'user_agent' => $request->header('User-Agent'),
            'url' => $request->fullUrl() . ' [' . $request->method() . ']',
            'location' => Location::get($request->ip()),
            'menseje' => $mensaje,
            'datos_permisos' => $datosPermiso
        ];

        if ($status == 'ok') {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->info("Eliminacion de permiso del sistema exitosa");
        } else {
            //Guardamos los datos en un archivo de log
            Log::withContext($arrayDatos)->warning("Eliminacion de permiso del sistema fallida");
        }
    }

    public function show()
    {
        return view('dashboard.configuraciones.sucursales');
    }

    public function sucursalesData()
    {
        return DataTables::of(Sucursale::all())
            ->addColumn('action', function ($sucursal) {
                return view('layouts.dashboard.partials.sucursales.botones', compact('sucursal'));
            })->make(true);
    }

    public function sucursalData($id)
    {
        //Retorna los datos de la sucursal en formato json
        return Response()->json(["success" => true, 'data' => Sucursale::findOrFail($id)]);
    }

    public function registraSucursal(Request $request)
    {
        $validaSucursal = Sucursale::where('sucursal', $request->sucursal)->count();

        if ($validaSucursal >= 1) {
            notify()->error('La sucursal ya se encuentra registrada', 'Registro de sucursal');
        } else {
            //Se obtiene los datos de la sucursal
            $sucursal = new Sucursale();
            $sucursal->sucursal = strtoupper($request->sucursal);
            $sucursal->nombre = strtoupper($request->nombre);
            $sucursal->tipo = $request->tipo;
            $sucursal->activa = $request->estado_suc;
            $sucursal->save();

            SucursalesController::setControlNumReceta(strtolower($request->sucursal));
            SucursalesController::creaDBSucursal(strtolower($request->sucursal));

            notify()->success('Sucursal registrada correctamente', 'Registro de sucursal');
        }
        return redirect()->route('configuraciones.sucursales');
    }


    private static function creaDBSucursal($sucursal)
    {
        //Sentencia para crear la tabla de registro de movimientos de la sucursal
        Schema::create('registro_movimientos_' . strtolower($sucursal), function (Blueprint $table) {
            $table->increments('id');
            $table->string('tipo_movimiento', 7);
            $table->string('fecha', 45);
            $table->string('hora', 45);
            $table->string('codigo_producto', 255);
            $table->string('producto', 255);
            $table->string('denominacion_dist', 45);
            $table->string('denominacion_gen', 45);
            $table->string('presentacion', 45);
            $table->string('laboratorio', 60)->default("SIN LABORATORIO");;
            $table->string('procedencia', 45);
            $table->string('nombre_pree', 45);
            $table->longText('domicilio');
            $table->string('cedula', 15);
            $table->string('factura', 15);
            $table->string('receta', 15);
            $table->integer('cantidad_adquirida');
            $table->integer('cantidad_vendida');
            $table->integer('saldo');
            $table->longText("observaciones");
            $table->timestamps();
        });

        Log::info("Base de datos de la sucural [" . $sucursal . "] creada");
    }

    private static function setControlNumReceta($sucursal)
    {
        $numeroReceta = new NumeroReceta();
        $numeroReceta->sucursal = strtolower($sucursal);
        $numeroReceta->numero_receta = 0;
        $numeroReceta->save();

        Log::info("Control del numero de receta de la sucursal [" . $sucursal . "] creada");
    }

    public function actulizaSucursal(Request $request)
    {
        //Recuperamos la sucursal a actualizar
        $id = $request->id_sucursal_edit;

        $validaSucursal = Sucursale::where('sucursal', $request->sucursal_edit)->where('id', '!=', $id)->count();

        if ($validaSucursal >= 1) {
            notify()->error('La sucursal ya se encuentra registrada', 'Actualizacion de sucursal');
        } else {
            //Se obtiene los datos de la sucursal
            $sucursal = Sucursale::findOrFail($id);
            $sucursal->sucursal = strtoupper($request->sucursal_edit);
            $sucursal->nombre = strtoupper($request->nombre_edit);
            $sucursal->tipo = $request->tipo_edit;
            $sucursal->activa = $request->estado_suc_edit;
            $sucursal->update();

            notify()->success('Sucursal actualizada correctamente', 'Actualizacion de sucursal');
        }
        return redirect()->route('configuraciones.sucursales');
    }
}
